// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : entityPlayer.cpp
// Date           : February 2008
//
// Description    : Refer to description in corresponding header.
//
// ///////////////////////////////////////////////////////////////////////////


#include "entityPlayer.h"
#include "Level.h"
#include "Application.h"




EntityPlayer::EntityPlayer() 
{

  m_Type = Dobbs::ENTITY_TYPE_PLAYER;

  m_CanBeCarried    = true;

  m_soundRunDelay   = 0;     // Delay since last run sound was played
  m_soundRun        = 0;     // ID of last run sound played
  m_JumpCount       = 0;     // Jump count since last touched ground
  m_isDropping      = false; // Indicates whether or not player is dropping
  m_PlayerState     = Dobbs::STATE_OK;
  m_pCannon         = NULL;
  m_pSub            = NULL;
  m_WarpIn          = false;
  m_WarpOut         = false;
  m_WarpTime        = 0.0f;

  SetRect( &m_CollisionRect, -14, -63, 14, 0 );

  // Load sprite and sound resources
  loadResources();

}



EntityPlayer::~EntityPlayer() 
{
  unloadResources();
}



void EntityPlayer::loadResources() 
{
  // Select sprite images based on 'character' value and adjust collision
  // offsets, as appropriate for each character.  Create action sprites
  // for left/right running, jumping and standing.
  m_DisplayOffset.x = 21;
  m_DisplayOffset.y = 63;

  // Create sounds for running, jumping, landing, and exploding.
  // NOTE:  There are 3 running sounds that are played back-to-back
  //        in random order to create a prolonged, but unrepetative
  //        sound of running.
  SetFrame( Dobbs::ANIM_PLAYER_STAND_LEFT );

}



void EntityPlayer::unloadResources() 
{

}



void EntityPlayer::RunSound( bool Running )
{

  if ( Running )
  {
    m_soundRunDelay = ( m_soundRunDelay + 1 ) % 10;
    if ( m_soundRunDelay == 0 )
    {
      char    Temp[20];
      wsprintfA( Temp, "Player.Run.%d", m_soundRun + 1 );
      g_App.PlaySound( Temp );
      int previous = m_soundRun;
      while ( previous == m_soundRun ) 
      { 
        m_soundRun = rand() % 3;
      }
    }
  }

}



void EntityPlayer::Drop( Level& aLevel ) 
{
  // Ignore if already in the air
  if ( !m_OnGround )
  {
    return;
  }
  if ( m_pCarriedBy )
  {
    m_pCarriedBy->DropOff( this );
  }

  // If onGround, then move down one pixel to force player to fall
  Move( aLevel, 0, 1, Dobbs::DIR_DOWN );

}



Dobbs::PlayerState EntityPlayer::GetState() 
{ 

  return m_PlayerState; 

}



void EntityPlayer::AdjustPhysics( int dSpeedGrass, int dSpeedGlue,
                                  int dSoundGrass, int dSoundGlue,
                                  int dGravDelay, float dGravAccel, float dGravMax,
                                  float dJumpInit, int dJumpMax ) 
{

  // Adjust character specifics by incrementals specified
  m_speedGrass     += dSpeedGrass;
  m_speedGlue      += dSpeedGlue;
  m_soundGrass     += dSoundGrass;
  m_soundGlue      += dSoundGlue;
  m_gravityDelay   += dGravDelay;
  m_gravityAccel   += dGravAccel;
  m_gravityMax     += dGravMax;

}



void EntityPlayer::HandleAnimation()
{

  if ( m_PlayerState != Dobbs::STATE_OK )
  {
    if ( m_PlayerState == Dobbs::STATE_IN_VEHICLE )
    {
      if ( m_Direction == Dobbs::DIR_LEFT )
      {
        SetFrame( Dobbs::ANIM_PLAYER_DROP_LEFT );
      }
      else
      {
        SetFrame( Dobbs::ANIM_PLAYER_DROP_RIGHT );
      }
    }
    return;
  }

  if ( m_Jumping )
  {
    if ( m_Direction == Dobbs::DIR_LEFT )
    {
      SetFrame( Dobbs::ANIM_PLAYER_JUMP_LEFT );
    }
    else
    {
      SetFrame( Dobbs::ANIM_PLAYER_JUMP_RIGHT );
    }
  }
  else if ( !m_OnGround )
  {
    if ( m_Direction == Dobbs::DIR_LEFT )
    {
      SetFrame( Dobbs::ANIM_PLAYER_DROP_LEFT );
    }
    else
    {
      SetFrame( Dobbs::ANIM_PLAYER_DROP_RIGHT );
    }
  }
  else
  {
    if ( m_MovedLastFrame )
    {
      if ( m_Direction == Dobbs::DIR_LEFT )
      {
        SetFrame( Dobbs::ANIM_PLAYER_RUN_LEFT );
      }
      else
      {
        SetFrame( Dobbs::ANIM_PLAYER_RUN_RIGHT );
      }
    }
    else
    {
      if ( m_Direction == Dobbs::DIR_LEFT )
      {
        SetFrame( Dobbs::ANIM_PLAYER_STAND_LEFT );
      }
      else
      {
        SetFrame( Dobbs::ANIM_PLAYER_STAND_RIGHT );
      }
    }
  }

}



void EntityPlayer::OnEvent( const EntityEvent Event, int Param1, int Param2, const std::string& TextParam, Entity* pEntity )
{

  switch ( Event )
  {
    /*
    case EE_BLOCKED_CARRIED_MOVE:
      // Tot!
      Die();
      break;
      */
    case EE_LAND:
      g_App.PlaySound( "Player.Land" );
      m_VelX = 0.0f;
      m_VelY = 0.0f;
      break;
    case EE_JUMP:
      g_App.PlaySound( "Player.Jump" );
      break;
    case EE_CARRIED:
      m_VelX = 0.0f;
      m_VelY = 0.0f;
      if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_CAR )
      {
        m_PlayerState = Dobbs::STATE_IN_VEHICLE;
        // force player into center of vehicle
        if ( Move( *m_pLevel, (float)( pEntity->GetX() - GetX() ), (float)( pEntity->GetY() - GetY() ), Dobbs::DIR_FORCED_MOVE ) != Dobbs::BLOCKED_NONE )
        {
          // cannot enter
          pEntity->DropOff( this );
        }
        else
        {
          if ( m_Direction == Dobbs::DIR_LEFT )
          {
            SetFrame( Dobbs::ANIM_PLAYER_DROP_LEFT );
          }
          else
          {
            SetFrame( Dobbs::ANIM_PLAYER_DROP_RIGHT );
          }
        }
      }
      break;
    case EE_DROPPED:
      if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_CAR )
      {
        m_PlayerState = Dobbs::STATE_OK;
      }
      else if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_SUB )
      {
        m_PlayerState = Dobbs::STATE_OK;
        m_Floating = false;
        m_CarryAgainDelay = 0.5f;
        m_pPreviousCarrier = NULL;
        m_pSub = NULL;
        Unhide();
        m_pLevel->Move( this, 0, -48, Dobbs::DIR_FORCED_MOVE );
        ForceJump( *m_pLevel, 600 );
      }
      break;
    case EE_DIE:
      m_pLevel->m_pGame->OnGameEvent( Dobbs::GE_PLAYER_DIED );
      m_pLevel->SpawnEntity( Dobbs::ENTITY_TYPE_PLAYER_EXPLOSION, 0, GetX(), GetY() );
      g_App.PlaySound( "Player.Explode" );
      break;
  }
  Entity::OnEvent( Event, Param1, Param2, TextParam, pEntity );

}



void EntityPlayer::SetPlayerState( Dobbs::PlayerState State )
{

  m_PlayerState = State;

}



void EntityPlayer::IntoCannon( Entity* pCannon )
{

  if ( ( m_CarryAgainDelay > 0.0f )
  &&   ( m_pPreviousCarrier == pCannon ) )
  {
    return;
  }

  //SetPosition( pCannon->GetX() + 4, pCannon->GetY() );
  SetPlayerState( Dobbs::STATE_IN_CANNON );
  m_Floating    = true;
  m_VelX        = 0.0f;
  m_VelY        = 0.0f;
  m_Jumping     = false;
  m_JumpCount   = 0;
  m_JumpVelocity  = 0.0f;
  m_FallSpeed   = 0;
  m_FallCount   = 0;

  m_UsingAngle  = true;
  m_Angle       = pCannon->Angle();

  m_pCannon     = pCannon;
  m_pCannon->SetFrame( Dobbs::ANIM_CANNON_FULL );
  Hide();

}



void EntityPlayer::ShootFromCannon()
{

  if ( m_pCannon == NULL )
  {
    return;
  }

  float     DX = 32.0f * sinf( m_Angle * 3.1415926f / 180.0f );
  float     DY = -32.0f * cosf( m_Angle * 3.1415926f / 180.0f );

  // 16,32 approximated from players center offset
  Move( *m_pLevel, DX + 16.0f, DY + 32.0f );

  SetPlayerState( Dobbs::STATE_OK );
  m_Floating    = false;
  m_VelX        = 50.0f * DX;
  m_VelY        = 50.0f * DY;
  m_CarryAgainDelay = 0.5f;

  m_UsingAngle  = false;

  m_pCannon->SetFrame( Dobbs::ANIM_CANNON );
  m_pPreviousCarrier = m_pCannon;
  m_pCannon     = NULL;
  Unhide();

  for ( int i = 0; i < 5; ++i )
  {
    Entity*   pExplosion = m_pLevel->SpawnEntity( Dobbs::ENTITY_TYPE_PARTICLE_EXPLOSION, 0, m_pPreviousCarrier->GetX() + 32, m_pPreviousCarrier->GetY() + 32 );

    float   CDX = rand() % 81 - 40.0f;
    float   CDY = rand() % 81 - 40.0f;

    CDX += DX * 13.0f;
    CDY += DY * 13.0f;

    pExplosion->SetVelX( CDX );
    pExplosion->SetVelY( CDY );
    pExplosion->SetLifeTime( 0.6f - ( rand() % 20 ) * 0.01f );
  }

  g_App.PlaySound( "Cannon" );

}



void EntityPlayer::IntoSub( Entity* pSub )
{

  if ( ( m_CarryAgainDelay > 0.0f )
  &&   ( m_pPreviousCarrier == pSub ) )
  {
    return;
  }
  if ( m_pSub == pSub )
  {
    return;
  }

  Hide();
  SetPlayerState( Dobbs::STATE_IN_SUB );
  m_Floating    = true;
  m_VelX        = 0.0f;
  m_VelY        = 0.0f;
  m_Jumping     = false;
  m_JumpCount   = 0;
  m_JumpVelocity  = 0.0f;
  m_FallSpeed   = 0;
  m_FallCount   = 0;
  m_FractX      = 0.0f;
  m_FractY      = 0.0f;

  m_pSub        = pSub;
  m_pSub->SetFrame( Dobbs::ANIM_SUB_R );
  m_pSub->Carry( this );

}



void EntityPlayer::ExitSub()
{

  if ( m_pSub == NULL )
  {
    return;
  }

  ForceJump( *m_pLevel, 600 );

  SetPlayerState( Dobbs::STATE_OK );
  m_Floating    = false;
  m_CarryAgainDelay = 0.5f;

  m_pSub->SetFrame( Dobbs::ANIM_SUB_EMPTY );
  m_pSub->DropOff( this );
  m_pPreviousCarrier = m_pSub;
  m_pSub             = NULL;
  Unhide();

}



void EntityPlayer::Render( int XOffset, int YOffset )
{

  if ( m_WarpIn )
  {
    YOffset += (int)( m_WarpTime * 600.0f );
  }
  if ( m_WarpOut )
  {
    YOffset += 600 - (int)( m_WarpTime * 600.0f );
  }
  if ( m_pCannon )
  {
    m_Angle = m_pCannon->Angle();
  }
  Entity::Render( XOffset, YOffset );

}



void EntityPlayer::UpdateTimed( Level& aLevel, float ElapsedTime )
{

  if ( m_PlayerState != Dobbs::STATE_WARPING )
  {
    Entity::UpdateTimed( aLevel, ElapsedTime );
  }

  if ( ( m_WarpIn )
  ||   ( m_WarpOut ) )
  {
    m_WarpTime -= ElapsedTime;
    if ( m_WarpTime <= 0.0f )
    {
      if ( m_WarpOut )
      {
        Hide();
      }
      else
      {
        m_DisplayOffset.x = 21;
        m_WarpIn = false;
        SetPlayerState( Dobbs::STATE_OK );
        SetFrame( Dobbs::ANIM_PLAYER_STAND_RIGHT );
      }
    }
  }

}



void EntityPlayer::Update( Level& aLevel )
{

  if ( m_PlayerState == Dobbs::STATE_WARPING )
  {
    return;
  }

  // slow shot speed
  if ( m_VelX != 0.0f )
  {
    m_VelX *= 0.9f;
    if ( fabs( m_VelX ) < 1.0f )
    {
      m_VelX = 1.0f;
    }
  }
  if ( m_VelY != 0.0f )
  {
    m_VelY *= 0.9f;
    if ( fabs( m_VelY ) < 1.0f )
    {
      m_VelY = 1.0f;
    }
  }

  Entity::Update( aLevel );

}



void EntityPlayer::WarpIn()
{

  m_WarpIn      = true;
  m_WarpTime    = 0.75f;
  m_pCarriedBy  = NULL;
  m_DisplayOffset.x = 11;

  SetFrame( Dobbs::ANIM_PLAYER_WARP );
  SetPlayerState( Dobbs::STATE_WARPING );

  g_App.PlaySound( "Player.Teleport" );

}



void EntityPlayer::WarpOut()
{

  m_WarpOut   = true;
  m_WarpTime  = 1.0f;
  m_DisplayOffset.x = 11;

  SetFrame( Dobbs::ANIM_PLAYER_WARP );
  SetPlayerState( Dobbs::STATE_WARPING );

  g_App.PlaySound( "Player.Teleport" );

}
